#
############################################################
# MODULE:    Handlers for MARC Description File
# VERSION:   1.0
# DATE:      November 17, 1997
#
# MULBERRY INTERNAL VERSION CONTROL:
# $Id: marcdesc.pl,v 1.4 1997/11/26 14:59:54 tkg Exp $
############################################################

############################################################
# SYSTEM:    MARC to tag-valid SGML converter set
#
# PURPOSE:   Process MARC Description File and build data structures
#            for use by the conversion program
#
# CONTAINS:  1) Initialise global variables
#            2) Declare handlers for elements in the MARC Description
#               File
#            3) Declare classes for data structures
#            4) Diagnostic subroutine
#
# PACKAGES REQUIRED:
#            1) David Megginson's SGMLS and SGMLS::Output packages
#            2) sgmlspm package
#
# CREATED FOR:
#            Network Development and MARC Standards Office
#            The Library of Congress
#
# ORIGINAL CREATION DATE:
#            November 1997
#
# CREATED BY:
#            Mulberry Technologies, Inc.
#            17 West Jefferson Street, Suite 207
#            Rockville, MD  20850
#            Phone:  301/315-9631
#            Fax:    301/315-8285
#            e-mail: info@mulberrytech.com
#            WWW:    http://www.mulberrytech.com
############################################################

############################################################
# Initalisation
$gMarcDesc = new SGMLSPL;

############################################################
# Global variables

# Hash relating Leader 06 and 07 data to document type
%gLdrToDoctype = ();

############################################################
# Declare handlers

SGMLSPL::sgml($gMarcDesc, '<DOCTYPE.SELECTOR>', sub {
    my $lElement = shift;

    $gLdrToDoctype{$lElement->attribute('LEADER.06.07')->value} =
	new DOCTYPE_SELECTOR(
			     $lElement->attribute('LEADER.06.07')->value,
			     $lElement->attribute('DOCTYPE')->value,
			     $lElement->attribute('FORMAT.TYPE')->value,
			     $lElement->attribute('RECORD.TYPE')->value);
});

SGMLSPL::sgml($gMarcDesc, '<DTD.GROUPS>', sub {
    my $lElement = shift;

    $lElement->set_attribute(new SGMLS_Attribute('Self',
						   'Perl',
						   new DTD_GROUPS));

    $lElement->attribute('Self')->value
	->set_doctype($lElement->attribute('DOCTYPE')->value);
});

SGMLSPL::sgml($gMarcDesc, '</DTD.GROUPS>', sub {
    my $lElement = shift;

    $gDtdGroups{$lElement->attribute('Self')->value->doctype} =
        $lElement->attribute('Self')->value->field_span;

});

SGMLSPL::sgml($gMarcDesc, '<FIELD.SPAN>', sub {
    my $lElement = shift;

    $lElement->parent->attribute('Self')->value->
	set_field_span(new FIELD_SPAN(
				      $lElement->attribute('START')->value,
				      $lElement->attribute('END')->is_implied ?
				      '' : $lElement->attribute('END')->value,
				      $lElement->attribute('LABEL')->value));
});

SGMLSPL::sgml($gMarcDesc, '<LDR.CLUSTER.GROUP>', sub {
    my $lElement = shift;

    $gLdrClusters{$lElement->attribute('KEY')->value} =
	new LDR_CLUSTERS(split(/\s+/,
			       $lElement->attribute('CLUSTERS')->value));
});

SGMLSPL::sgml($gMarcDesc, '<POSDEF>', sub {
    my $lElement = shift;

    $lElement->set_attribute(new SGMLS_Attribute('Self',
						   'Perl',
						   new POSDEF_FIELD));
    $lElement->attribute('Self')->value->
	set_key($lElement->attribute('KEY')->value);
});

SGMLSPL::sgml($gMarcDesc, '</POSDEF>', sub {
    my $lElement = shift;

    $gControlFields{$lElement->attribute('Self')->value->marc_tag} =
	$lElement->attribute('Self')->value;

});

SGMLSPL::sgml($gMarcDesc, '<NOINDSF>', sub {
    my $lElement = shift;

    $lElement->set_attribute(new SGMLS_Attribute('Self',
						   'Perl',
						   new NOINDSF_FIELD));
});

SGMLSPL::sgml($gMarcDesc, '</NOINDSF>', sub {
    my $lElement = shift;

    $gControlFields{$lElement->attribute('Self')->value->marc_tag} =
	$lElement->attribute('Self')->value;
});

SGMLSPL::sgml($gMarcDesc, '<KEY.CLUSTER.GROUP>', sub {
    my $lElement = shift;
    my @lClusters = split(/\s+/, $lElement->attribute('CLUSTERS')->value);
    my $lFieldLength =	$lClusters[$#lClusters];

    # The last cluster may be a range, so maybe the length is the last
    # character position in the range.
    if ($lFieldLength =~ /\d+-(\d+)/) {
	$lFieldLength = $1;
    }

    $lElement->parent->attribute('Self')->value
	->set_clusters(new POSDEF_CLUSTERS
		       ($lElement->attribute('KEY')->value,
			$lElement->attribute('SUBTYPE')->value,
			# Field length is 0 to the max number, therefore
			# increment the max number to get the length.
			++$lFieldLength,
			@lClusters));

});

SGMLSPL::sgml($gMarcDesc, '<MARC.TAG>', sub {
    push_output 'string';
});

SGMLSPL::sgml($gMarcDesc, '</MARC.TAG>', sub {
    my $lElement = shift;

    $lElement->parent->attribute('Self')->value->set_marc_tag(pop_output);
});

############################################################


package DOCTYPE_SELECTOR;

					# Constructor.
sub new {
    my ($pSelf, $pLeader_06_07, $pDoctype, $pFormat, $pRecordType) = (@_);

    $pDoctype =~ tr/A-Z/a-z/;

    return bless [
		  $pLeader_06_07,
		  $pDoctype,
                  $pFormat,
		  $pRecordType
		  ];
}

				# Set the Leader character positions 06 and 07
sub set_leader_06_07 {
    my ($self,$leader_06_07) = @_;
    $self->[0] = $leader_06_07;
}

				# Set the doctype
sub set_doctype {
    my ($self,$doctype) = @_;
    ($self->[1] = $doctype) =~ tr/A-Z/a-z/;
}

				# Set the format type
sub set_format_type {
    my ($self,$format_type) = @_;
    $self->[2] = $format_type;
}

				# Set the record type
sub set_record_type {
    my ($self,$record_type) = @_;
    $self->[3] = $record_type;
}

					# Accessors.
sub leader_06_07 { return $_[0]->[0]; }
sub doctype { return $_[0]->[1]; }
sub format_type { return $_[0]->[2]; }
sub record_type { return $_[0]->[3]; }

############################################################

package FIELD_SPAN;

					# Constructor.
sub new {
    my ($pSelf, $pStart, $pEnd, $pLabel) = (@_);

    return bless [
		  $pStart,
		  $pEnd,
		  $pLabel
		  ];
}

				# Set the start of range
sub set_start {
    my ($self,$start) = @_;
    $self->[0] = $start;
}

				# Set the end of range
sub set_end {
    my ($self,$end) = @_;
    $self->[1] = $end;
}

				# Set the range label
sub set_label {
    my ($self,$label) = @_;
    $self->[2] = $label;
}

					# Accessors.
sub start { return $_[0]->[0]; }
sub end { return $_[0]->[1]; }
sub label { return $_[0]->[2]; }

############################################################

package DTD_GROUPS;

					# Constructor.
sub new {
    my ($pSelf, $pDoctype) = (@_);

    $pDoctype =~ tr/A-Z/a-z/;

    return bless [
		  $pDoctype,
                  {}
		  ];
}

				# Set the doctype
sub set_doctype {
    my ($self,$doctype) = @_;
    ($self->[0] = $doctype) =~ tr/A-Z/a-z/;
}

				# Set the field span
sub set_field_span {
    my ($self, $field_span) = @_;
    $self->field_span->{$field_span->start} = $field_span;
}

					# Accessors.
sub doctype { return $_[0]->[0]; }
sub field_span { return $_[0]->[1]; }

############################################################

package LDR_CLUSTERS;

					# Constructor.
sub new {
    my ($pSelf, $pKey, @pClusters) = (@_);

    return bless [
		  $pKey,
                  @pClusters,
		  ];
}

				# Set the key
sub set_key {
    my ($self,$key) = @_;
    $self->[0] = $key;
}

				# Push a cluster value
sub set_clusters {
    my ($self, $clusters) = @_;
    push(@{$self->[1]}, split(/\s+/, $cluster));
}

					# Accessors.
sub key { return $_[0]->[0]; }
sub clusters { return $_[0]->[1]; }

############################################################

package POSDEF_CLUSTERS;

					# Constructor.
sub new {
    my ($pSelf, $pKey, $pSubtype, $pFieldLength, @pClusters) = (@_);

    return bless [
		  $pKey,
                  $pSubtype,
		  $pFieldLength,
		  \@pClusters,
		  ];
}

				# Set the key
sub set_key {
    my ($self,$key) = @_;
    $self->[0] = $key;
}

				# Set the subtype
sub set_subtype {
    my ($self,$subtype) = @_;
    $self->[1] = $subtype;
}

				# Push a cluster value
sub set_clusters {
    my ($self, $cluster) = @_;
    push(@{$self->[2]}, split(/\s+/, $cluster));
}

					# Accessors.
sub key { return $_[0]->[0]; }
sub subtype { return $_[0]->[1]; }
sub field_length { return $_[0]->[2]; }
sub clusters { return $_[0]->[3]; }

############################################################

package POSDEF_FIELD;

					# Constructor.
sub new {
    my ($pSelf, $pMarcTag, $pKey, %pClusters) = (@_);

    return bless [
		  $main::cPositionallyDefinedField,
		  $pMarcTag,
		  $pKey,
		  %pClusters,
		  ];
}

				# Set the MARC tag
sub set_marc_tag {
    my ($self, $marc_tag) = @_;
    $self->[1] = $marc_tag;
}

				# Set the description of what to use as key
sub set_key {
    my ($self, $key) = @_;
    $self->[2] = $key;
}

sub set_clusters {
    my ($self, $clusters) = @_;
    $self->[3]->{$clusters->key} = $clusters;
}

					# Accessors.
sub field_type { return $_[0]->[0]; }
sub marc_tag { return $_[0]->[1]; }
sub key { return $_[0]->[2]; }
sub clusters { return $_[0]->[3]; }

############################################################

package NOINDSF_FIELD;

					# Constructor.
sub new {
    my ($pSelf, $pMarcTag) = (@_);

    return bless [
		  $main::cNoIndicatorsOrSubfields,
		  $pMarcTag,
		  ];
}

				# Set the MARC tag
sub set_marc_tag {
    my ($self,$marc_tag) = @_;
    $self->[1] = $marc_tag;
}

					# Accessors.
sub field_type { return $_[0]->[0]; }
sub marc_tag { return $_[0]->[1]; }

############################################################
package main;

############################################################
# &DumpMarcDesc()
# Dump the data in the MARC Description File in a format that
# someone can puzzle out what's happening, not necessarily in
# a format you'd feed to another program
sub DumpMarcDesc {

    foreach (sort(keys(%gLdrToDoctype))) {
	print "leader.06.07:" . $gLdrToDoctype{$_}->leader_06_07 . ":\n";
	print "doctype:" . $gLdrToDoctype{$_}->doctype . "\n";
	print "format.type:" . $gLdrToDoctype{$_}->format_type . ":\n";
	print "record.type:" . $gLdrToDoctype{$_}->record_type . ":\n";
    }

    foreach $lDoctype (sort(keys(%gDtdGroups))) {

	print "doctype:$lDoctype:\n";

	foreach (sort(keys(%{$gDtdGroups{$lDoctype}}))) {
	    print "start:" . ${$gDtdGroups{$lDoctype}}{$_}->start . "\n";
	    print "end:" . ${$gDtdGroups{$lDoctype}}{$_}->end . "\n";
            print "label:" . ${$gDtdGroups{$lDoctype}}{$_}->label . "\n";
        }
    }

    foreach $lKey (sort(keys(%gLdrClusters))) {
	print "key:" . $lKey . ":\n";
	print "key:" . $gLdrClusters{$lKey}->key . ":\n";
        print "clusters:" ;

        foreach (@{$gLdrClusters{$lKey}}) {
            print "$_:";
        }

	print "\n";
    }

    foreach $lKey (sort(keys(%gControlFields))) {
	print "marc.tag:" . $lKey . ":\n";
	print "type:" . $gControlFields{$lKey}->field_type . ":\n";

        if ($gControlFields{$lKey}->field_type =~
    	    /$cPositionallyDefinedField/o) {

	    print "key position:" . $gControlFields{$lKey}->key . ":\n";

	    foreach $lClusterKey (sort(keys(%{$gControlFields{$lKey}->
						  clusters}))) {
		print "key:" . $lClusterKey . ":\n";
		print "Clusters:" . $gControlFields{$lKey}->clusters->
					{$lClusterKey} . ":\n";
		print "subtype:" . $gControlFields{$lKey}->clusters->
					{$lClusterKey}->subtype . ":\n";
		print "field_length:" . $gControlFields{$lKey}->clusters->
					{$lClusterKey}->field_length . ":\n";
		print "Clusters:" . @{$gControlFields{$lKey}
				      ->clusters
					  ->{$lClusterKey}->
					      clusters} . ":\n";

		foreach $lCluster (@{$gControlFields{$lKey}
				     ->clusters->{$lClusterKey}->clusters}) {
		    print "$lCluster:\n";
		}
	    }
	}
    }
}

############################################################
1;
